<?php

namespace NeuronAI\Tools;

class ArrayProperty implements ToolPropertyInterface
{
    protected PropertyType $type = PropertyType::ARRAY;

    public function __construct(
        protected string $name,
        protected string $description,
        protected bool $required = false,
        protected array $items = [],
    ) {
    }

    public function jsonSerialize(): array
    {
        return [
            'name' => $this->name,
            'description' => $this->description,
            'type' => $this->type->value,
            'items' => $this->getJsonSchema(),
            'required' => $this->required,
        ];
    }

    public function getRequiredProperties(): array
    {
        return array_values(\array_filter(\array_map(function (ToolPropertyInterface $property) {
            return $property->isRequired() ? $property->getName() : null;
        }, $this->items)));
    }

    public function getJsonSchema(): array
    {
        $schema = [
            'type' => $this->type->value,
            'description' => $this->description,
        ];

        $items = \array_reduce($this->items, function (array $carry, ToolPropertyInterface $property) {
            $carry[$property->getName()] = $property->getJsonSchema();
            return $carry;
        }, []);

        if (!empty($items)) {
            $schema['items'] = [
                'type' => 'object',
                'properties' => $items,
                'required' => $this->getRequiredProperties(),
            ];
        }

        return $schema;
    }

    public function isRequired(): bool
    {
        return $this->required;
    }

    public function getName(): string
    {
        return $this->name;
    }

    public function getType(): PropertyType
    {
        return $this->type;
    }

    public function getDescription(): string
    {
        return $this->description;
    }

    public function getItems(): array
    {
        return $this->items;
    }
}
